#include "EventStatus.h"

/*===========================================================================*/
/* SORENSEN DCS/DLM M9                                                       */
/* LabWindows/CVI Instrument Driver                                          */
/* Original Release:                                                         */
/* By:  R. Poyner                                                            */
/* PH:  858-450-0085 x2431                                                   */
/* Fax: 858-678-4441                                                         */
/*                                                                           */
/* Modification History: None                                                */
/*===========================================================================*/

#include <visa.h>
#include <userint.h>
#include <ansi_c.h>
#include <formatio.h>
#include <string.h>
#include "SNDCSDLM.h"

#define SNDCSDLM_REVISION   "V1.02, 06/2005, CVI 6.0.0"   /* Driver Revision */
#define BUFFER_SIZE         128L                          /* File I/O buffer */

#define DEMO_IF			0
#define GPIB_IF			1
#define RS232_IF		2
#define ENET_IF			3

#define LOCK                                                      \
    if ((SNDCSDLM_status = viLock(instrSession,					  \
                                  VI_EXCLUSIVE_LOCK,			  \
                                  THREAD_LOCK_TIMEOUT,			  \
                                  NULL,NULL)) < 0)                \
        return SNDCSDLM_status;                                   \

#define UNLOCK                                                    \
{                                                                 \
    if((unlockStatus = viUnlock(instrSession)) < 0)               \
        return unlockStatus;                                      \
    return SNDCSDLM_status;                                       \
}

/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* SNDCSDLM_stringValPair is used in the SNDCSDLM_errorMessage function          */
/* SNDCSDLM_statusDataRanges is used to track session dependent status & ranges*/
/*===========================================================================*/
typedef struct  SNDCSDLM_stringValPair
{
   ViStatus stringVal;
   ViString stringName;
}  SNDCSDLM_tStringValPair;
 

typedef struct SNDCSDLM_statusDataRanges *SNDCSDLM_instrRange;

static short int		user_if;  

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean SNDCSDLM_invalidViBooleanRange (ViBoolean val);
ViBoolean SNDCSDLM_invalidViInt16Range   (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean SNDCSDLM_invalidViInt32Range   (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean SNDCSDLM_invalidViUInt8Range   (ViUInt8 val, ViUInt8 min, ViUInt8 max);
ViBoolean SNDCSDLM_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean SNDCSDLM_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean SNDCSDLM_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean SNDCSDLM_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  SNDCSDLM_initCleanUp           (ViSession openRMSession,
                                          ViPSession openInstrSession,
                                          ViStatus currentStatus);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_init (ViRsrc     resourceName,
                                ViBoolean  IDQuery,
                                ViBoolean  resetDevice,
                                ViPSession instrSession)
{
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViSession	rmSession = 0;
    ViUInt32	retCnt = 0;
    ViInt16		i;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		tmpBuffer[10];

    /*- Open instrument session ---------------------------------------------*/
    user_if = GPIB_IF;
    if ((SNDCSDLM_status = viOpenDefaultRM (&rmSession)) < 0)
    {
    	sprintf(rdBuffer,"viOpenDefaultRM - %08X",SNDCSDLM_status);
		MessagePopup("SNDCSDLM Error",rdBuffer);
        return SNDCSDLM_status;
    }

    if ((SNDCSDLM_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0)
    {
    	sprintf(rdBuffer,"viOpen - %08X",SNDCSDLM_status);
		MessagePopup("SNDCSDLM Error",rdBuffer);
        viClose (rmSession);
        return SNDCSDLM_status;
    }
	viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 7000);

	/* set up serial port if selected */
	if (strstr (resourceName, "ASRL") != NULL)
	{
		user_if = RS232_IF;
		viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 19200);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_NONE);
		viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_TERMCHAR);
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\r');
	}
	else if ((resourceName, "TCPIP") != NULL)
	{
		user_if = ENET_IF;
		/*
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_TRUE);
		viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, '\n');
		*/
	}
	
    /*- Identification Query ------------------------------------------------*/
    if (IDQuery)
    {
		if ((SNDCSDLM_status = SNDCSDLM_idQuery (*instrSession,"1",rdBuffer)) < VI_SUCCESS)
            return SNDCSDLM_initCleanUp (rmSession, instrSession, SNDCSDLM_status);

		/* get and compare first eight bytes from *idn? response */
		for (i = 0; i < 8; ++i)
			tmpBuffer[i] = rdBuffer[i];
		tmpBuffer[8] = '\0';
		StringUpperCase (tmpBuffer);
		if (strcmp(tmpBuffer,"SORENSEN") != 0)
            return SNDCSDLM_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
    }
        
    /*- Reset instrument ----------------------------------------------------*/
    if (resetDevice)
    {
        if ((SNDCSDLM_status = SNDCSDLM_reset (*instrSession, "1")) != VI_SUCCESS)
            return SNDCSDLM_initCleanUp (rmSession, instrSession, SNDCSDLM_status);
    }       
          
    return SNDCSDLM_status;
}

/*===========================================================================*/
/* Function: GenerateDCVolt                                                  */
/* Purpose:  This function sets the output voltage to the specified value.   */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_GenerateDCVolt (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: GenerateDCCurr                                                  */
/* Purpose:  This function sets the output current to the specified value.   */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_GenerateDCCurr (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: GenerateDCVRamp                                                 */
/* Purpose:  This function initates a output voltage ramp from the present   */
/*           output voltage to the specified output voltage in the specified */
/*           time on the specified channel. Time is in seconds.              */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_GenerateDCVRamp  (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViReal64       voltage,
                                            ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:RAMP ");
    sprintf(tmpBuffer,"%f ",voltage);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}
                                          
/*===========================================================================*/
/* Function: GenerateDCCRamp                                                 */
/* Purpose:  This function initates a output Current ramp from the present   */
/*           output voltage to the specified output current in the specified */
/*           time on the specified channel. Time is in seconds.              */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_GenerateDCCRamp  (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViReal64       current,
                                            ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT:RAMP ");
    sprintf(tmpBuffer,"%f ",current);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: OutState                                                       */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_OutState (ViSession      instrSession,
                                    ViChar _VI_FAR channelName[],
                                    ViBoolean      state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":STATE ");
    sprintf(tmpBuffer,"%d",state);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
    
}

/*===========================================================================*/
/* Function: MeasureDCVolt                                                   */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_MeasureDCVolt (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViReal64       *voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
	if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	UNLOCK
}

/*===========================================================================*/
/* Function: MeasureDCCurr                                                   */
/* Purpose:  This function reads the output voltage.                         */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_MeasureDCCurr (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: Mode                                                            */
/* Purpose:  This function returns the operating mode of the power supply.   */
/*           (voltage mode = 0, current mode = 1)                            */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ModeQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViInt16        *mode)
{
    ViStatus		SNDCSDLM_status,unlockStatus;
	unsigned char	protCondReg  = 0;
	ViByte			statusMessage[BUFFER_SIZE];

	LOCK
	
	if ((SNDCSDLM_status = SNDCSDLM_ProtCondQuery(instrSession, channelName, &protCondReg, statusMessage)) < 0)
	{
		*mode = -1;
		UNLOCK;
	}

	if (protCondReg & SNDCSDLM_PE_CURRMODE)
		*mode = 1;
	else
		*mode = 0;
		
	UNLOCK
}

/*===========================================================================*/
/* Function: VoltMode                                                        */
/* Purpose:  This function configures the power supply to shutdown if the    */
/*           power supply switches from voltage mode to current mode.        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_VoltMode (ViSession      instrSession,
                                    ViChar _VI_FAR channelName[],
                                    ViBoolean      serviceRequest)
                                
{
    ViStatus		SNDCSDLM_status,unlockStatus;
	unsigned char	mask;

	LOCK
	
	mask =  SNDCSDLM_PE_CURRMODE | SNDCSDLM_PE_FOLD;
	
	/* program foldback */
	if ((SNDCSDLM_status = SNDCSDLM_ConfigFoldback (instrSession, channelName, SNDCSDLM_FB_CURRMODE)) < 0)
		UNLOCK
		
	/* program protection event enable mask for foldback and current mode */
	if ((SNDCSDLM_status = SNDCSDLM_ConfigProtEventEnable (instrSession, channelName, mask)) < 0)
		UNLOCK
		
	if (serviceRequest)
	{
		/* program service request enable mask */
		SNDCSDLM_status = SNDCSDLM_ConfigSerReqEnable (instrSession, channelName, SNDCSDLM_SB_SRQ);
	}
	
	UNLOCK
}

/*===========================================================================*/
/* Function: CurrMode                                                        */
/* Purpose:  This function configures the power supply to shutdown if the    */
/*           power supply switches from current mode to voltage mode.        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_CurrMode (ViSession      instrSession,
                                    ViChar _VI_FAR channelName[],
                                    ViBoolean      serviceRequest)
                                
{
    ViStatus	SNDCSDLM_status,unlockStatus;
	unsigned char	mask;

	LOCK
	
	mask =  SNDCSDLM_PE_VOLTMODE | SNDCSDLM_PE_FOLD;
	
	/* program foldback */
	if ((SNDCSDLM_status = SNDCSDLM_ConfigFoldback (instrSession, channelName, SNDCSDLM_FB_VOLTMODE)) < 0)
		UNLOCK
		
	/* program protection event enable mask */
	if ((SNDCSDLM_status = SNDCSDLM_ConfigProtEventEnable (instrSession, channelName, mask)) < 0)
		UNLOCK
		
	if (serviceRequest)
	{
		/* program service request enable mask */
		SNDCSDLM_status = SNDCSDLM_ConfigSerReqEnable (instrSession, channelName, SNDCSDLM_SB_SRQ);
	}
	
	UNLOCK    
}

/*===========================================================================*/
/* Function: NormalMode                                                      */
/* Purpose:  This function configures the power supply operate in both       */
/*           current mode and voltage mode. All protection events and        */
/*           service request enables are cleared.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_NormalMode (ViSession      instrSession,
                                      ViChar _VI_FAR channelName[])
{
    ViStatus	SNDCSDLM_status,unlockStatus;

	LOCK
	
	/* program foldback */
	if ((SNDCSDLM_status = SNDCSDLM_ConfigFoldback (instrSession, channelName, SNDCSDLM_FB_NORMAL)) < 0)
		UNLOCK
		
	/* program protection event enable mask */
	if ((SNDCSDLM_status = SNDCSDLM_ConfigProtEventEnable (instrSession, channelName, 0)) < 0)
		UNLOCK
		
	/* program service request enable mask */
	SNDCSDLM_status = SNDCSDLM_ConfigSerReqEnable (instrSession, channelName, 0);

	UNLOCK    
}

/*===========================================================================*/
/* Function: IsolationRelay                                                  */
/* Purpose:  This function turns on/off the m9 external user connector       */
/*           isolation relay.                                                 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_IsolationRelay (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViBoolean      state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":ISOLATION ");
    sprintf(tmpBuffer,"%d",state);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);
    
    UNLOCK
}

/*===========================================================================*/
/* Function: SenseRelay                                                      */
/* Purpose:  This function turns on/off the m9 external user connector sense */
/*           relay.															 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_SenseRelay (ViSession      instrSession,
                                      ViChar _VI_FAR channelName[],
                                      ViBoolean      state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":SENSE ");
    sprintf(tmpBuffer,"%d",state);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);
    
    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDCVolt                                                    */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage of the specified value. A software trigggger     */
/*           must be sent to the power supply to implement this sequence.    */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigDCVolt (ViSession      instrSession,
                                        ViChar _VI_FAR channelName[],
                                        ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:TRIGGERED ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}
                                          
/*===========================================================================*/
/* Function: ConfigDCCurr                                                    */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage of the specified value. A software trigggger     */
/*           must be sent to the power supply to implement this sequence.    */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigDCCurr (ViSession      instrSession,
                                        ViChar _VI_FAR channelName[],
                                        ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT:TRIGGERED ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDCVRamp                                                   */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output voltage ramp from the present output voltage to the      */
/*           specified voltage in the specified time on the specified        */
/*           channel. A software trigger must be sent to the power supply to */
/*           implement this sequence.                                        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigDCVRamp (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViReal64       voltage,
                                         ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:RAMP:TRIGGERED ");
    sprintf(tmpBuffer,"%f ",voltage);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}
                                          
/*===========================================================================*/
/* Function: ConfigDCCRamp                                                   */
/* Purpose:  This function configures the power supply for a triggered       */
/*           output current ramp from the present output current to the      */
/*           specified current in the specified time on the specified        */
/*           channel. A software trigger must be sent to the power supply to */
/*           implement this sequence. Time is in seconds.                    */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigDCCRamp (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViReal64       current,
                                         ViReal64       time_sec)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT:RAMP:TRIGGERED ");
    sprintf(tmpBuffer,"%f ",current);
    strcat(cmdBuffer,tmpBuffer);
    sprintf(tmpBuffer,"%f",time_sec);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigOVP                                                       */
/* Purpose:  This function sets the output overvoltage protection to the     */
/*           specified value.                                                */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigOVP (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViReal64       voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:PROTECTION ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSoftLimits                                                */
/* Purpose:  This function sets the soft voltage and current programming     */
/*           limits to the specified values.                                 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigSoftLimits (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViReal64       voltage,
                                            ViReal64       current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:LIMIT ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT:LIMIT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigPowerON                                                   */
/* Purpose:  This function configures the output power on values for the     */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigPowerOn (ViSession		instrSession,
                                         ViChar _VI_FAR	channelName[],
                                         ViReal64		voltage,
                                         ViReal64		current,
                                         ViReal64		ovp)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":INITIAL:VOLTAGE ");
    sprintf(tmpBuffer,"%f",voltage);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":INITIAL:CURRENT ");
    sprintf(tmpBuffer,"%f",current);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* build and send SCPI command string - ovp */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":INITIAL:VOLTAGE:PROTECTION ");
    sprintf(tmpBuffer,"%f",ovp);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);

    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - unlock */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":UNLOCK \"6867\"");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build  and send SCPI command string - store */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":STORE");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* build and send SCPI command string - lock */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":LOCK");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigFoldback                                                  */
/* Purpose:  This function configures the foldback mode if any. Three modes  */
/*           are normal, constant current, and constant voltage.             */
/*           See header file for bit definitions.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigFoldback (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViInt16        mode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":PROTECTION:FOLD ");
    sprintf(tmpBuffer,"%d",mode);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigProtDelay                                                 */
/* Purpose:  This function configures the programmable delay which is        */
/*           executed prior to reporting any output protection conditions.   */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigProtDelay (ViSession instrSession,
                                   ViChar    channelName[],
                                   ViReal64  delay)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":PROTECTION:DELAY ");
    sprintf(tmpBuffer,"%f",delay);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigVoltAve                                                   */
/* Purpose:  This function configures the number of current measurents to be */
/*           averaged. 1-5                                                   */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigVoltAve (ViSession instrSession,
                                         ViChar    channelName[],
                                         ViInt16   average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:AVERAGE ");
    sprintf(tmpBuffer,"%d",average);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigCurrAve                                                   */
/* Purpose:  This function configures the number of current measurents to be */
/*           averaged. 1-5                                                   */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigCurrAve (ViSession instrSession,
                                         ViChar    channelName[],
                                         ViInt16   average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"MEASURE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT:AVERAGE ");
    sprintf(tmpBuffer,"%d",average);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigProtEventEnable                                           */
/* Purpose:  This function configures the protection event status enable     */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigProtEventEnable (ViSession      instrSession,
                                                 ViChar _VI_FAR channelName[],
                                                 unsigned char  protectionEventEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":PROTECTION:ENABLE ");
    sprintf(tmpBuffer,"%d",protectionEventEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigStandEventEnable                                          */
/* Purpose:  This function configures the standard event status enable       */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigStandEventEnable (ViSession      instrSession,
											      ViChar _VI_FAR channelName[],
                                                  unsigned char  standardEventEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESE");
    strcat(cmdBuffer,channelName);
    sprintf(tmpBuffer," %d",standardEventEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSerReqEnable                                              */
/* Purpose:  This function configures the service request enable             */
/*           register. See header file for bit definitions.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigSerReqEnable (ViSession      instrSession,
                                              ViChar _VI_FAR channelName[],
                                              unsigned char  serviceRequestEnable)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"*SRE");
    strcat(cmdBuffer,channelName);
    sprintf(tmpBuffer," %d",serviceRequestEnable);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputVoltQuery                                                */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_OutputVoltQuery  (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViReal64       *voltage)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputCurrQuery                                                */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_OutputCurrQuery  (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

	/* get returnd query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: OutputOVPQuery                                                  */
/* Purpose:  This function returns the programmed output voltage value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_OutputOVPQuery (ViSession      instrSession,
                                           ViChar _VI_FAR channelName[],
                                           ViReal64       *OVP)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:PROTECTION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get return query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *OVP = atof(rdBuffer);

    UNLOCK
}
                                        

/*===========================================================================*/
/* Function: OutStateQuery                                                   */
/* Purpose:  This function returns the output state, 1 = ON.                 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_OutStateQuery (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViBoolean      *state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":STATE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *state = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: IsoRelayQuery                                                   */
/* Purpose:  This function returns isolation relay state.                    */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_IsoRelayQuery (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViBoolean      *state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":ISOLATION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *state = atoi(rdBuffer);

    UNLOCK
    
}

/*===========================================================================*/
/* Function: IsoRelayQuery                                                   */
/* Purpose:  This function returns sense relay state.                        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_SenseRelayQuery (ViSession      instrSession,
                                           ViChar _VI_FAR channelName[],
                                           ViBoolean      *state)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":SENSE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *state = atoi(rdBuffer);

    UNLOCK
    
}

/*===========================================================================*/
/* Function: SoftLimitsQuery                                                 */
/* Purpose:  This function returns the voltage and current soft limits.      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_SoftLimitsQuery  (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViReal64       *voltage,
                                            ViReal64       *current)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:LIMIT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	/* build and send SCPI command string - current limit */
    strcpy(cmdBuffer,"SOURCE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT:LIMIT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: PowerONQuery                                                    */
/* Purpose:  This function returns the output power on values for the        */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_PowerOnQuery  (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViReal64       *voltage,
                                         ViReal64       *current,
                                         ViReal64       *ovp)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];
    
	LOCK
	
	/* build and send SCPI command string - power on voltage */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":INITIAL:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *voltage = atof(rdBuffer);

	/* build and send SCPI comand string - power on current */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":INITIAL:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *current = atof(rdBuffer);

	/* build and send SCPI command string - power on ovp */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":INITIAL:VOLTAGE:PROTECTION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *ovp = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: protDelayQuery                                                  */
/* Purpose:  This function returns the protection condition delay value      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_protDelayQuery (ViSession instrSession,
                                          ViChar    channelName[],
                                          ViReal64  *delay)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"OUTPUT");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":PROTECTION:DELAY?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *delay = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: measVoltAveQuery                                                */
/* Purpose:  This function returns the number of voltage readings that will  */
/*           be averaged upon receipt of a voltage measurent command for the */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_measVoltAveQuery (ViSession instrSession,
                                            ViChar    channelName[],
                                            ViInt16   *average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"MEASURE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":VOLTAGE:AVERAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *average = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: measCurrAveQuery                                                */
/* Purpose:  This function returns the number of current readings that will  */
/*           be averaged upon receipt of a current measurent command for the */
/*           specified channel.                                              */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_measCurrAveQuery (ViSession instrSession,
                                            ViChar    channelName[],
                                            ViInt16   *average)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"MEASURE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":CURRENT:AVERAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *average = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: SendSWTrigger                                                   */
/* Purpose:  This function sends a software trigger to the specified         */
/*           channel. Setting the channel to zero causes a Group Execute     */
/*           Trigger for all power supplies that have been previously        */
/*           configured for triggered outputs.                               */
/*           Trigger Types: 1 = Voltage, 2 = Current, 3 = Both, 4 = Ramp     */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_SendSWTrigger (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViInt16        type)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
	/* test for voltage, current, or both trigger type */
    if ((type > 0) && (type < 4))
    {
		/* build SCPI command string */
	    strcpy(cmdBuffer,"TRIGGER");
    	strcat(cmdBuffer,channelName);
	    strcat(cmdBuffer,":TYPE ");
	    sprintf(tmpBuffer,"%d",type);
	    strcat(cmdBuffer,tmpBuffer);
	}
	/* test for ramp trigger type */
	else if (type == 4)
    {
		/* build SCPI command string */
	    strcpy(cmdBuffer,"TRIGGER");
    	strcat(cmdBuffer,channelName);
	    strcat(cmdBuffer,":RAMP");
	}
	else
		return VI_ERROR_INV_PARAMETER;
	
	/* send SCPI command string */
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    UNLOCK
}

/*===========================================================================*/
/* Function: Abort Trigger                                                   */
/* Purpose:  This function clears all previously configured voltage and      */
/*           current triggers for the specified channel.                     */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_AbortTrigger (ViSession      instrSession,
                                        ViChar _VI_FAR channelName[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"TRIGGER");
   	strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":ABORT");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: StatusByteQuery                                                 */
/* Purpose:  This function returns the instruments status byte and a message */
/*           indicating which bits are active.                               */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_StatusByteQuery (ViSession      instrSession,
                                           ViChar _VI_FAR channelName[],
                                           unsigned char  *statusByte,
                                           ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*STB");
   	strcat(cmdBuffer,channelName);
   	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*statusByte = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from status byte bits, bits 0,3,7 are NOT USED */
	if (*statusByte == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*statusByte);
		if (*statusByte & SNDCSDLM_SB_PROTEVENT)
			strcat(rdBuffer,"Protection Event, ");
		if (*statusByte & SNDCSDLM_SB_ERRORQUERY)
			strcat(rdBuffer,"Error Event Message, ");
		if (*statusByte & SNDCSDLM_SB_MAV)
			strcat(rdBuffer,"Message Available, ");
		if (*statusByte & SNDCSDLM_SB_STANDEVENT)
			strcat(rdBuffer,"Event Status, ");
		if (*statusByte & SNDCSDLM_SB_SRQ)
			strcat(rdBuffer,"Request Service");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: StandEventQuery                                                 */
/* Purpose:  This function returns the instruments standard event status     */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_StandEventQuery (ViSession      instrSession,
                                           ViChar _VI_FAR channelName[],
                                           unsigned char  *standardEventStatusRegister,
                                           ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESR");
   	strcat(cmdBuffer,channelName);
   	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*standardEventStatusRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits, bits 1,6 are NOT USED */
	if (*standardEventStatusRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*standardEventStatusRegister);
		if (*standardEventStatusRegister & SNDCSDLM_SE_OPC)
			strcat(rdBuffer,"Operation Complete, ");
		if (*standardEventStatusRegister & SNDCSDLM_SE_ERRORQUERY)
			strcat(rdBuffer,"Query Error, ");
		if (*standardEventStatusRegister & SNDCSDLM_SE_DEVICEERROR)
			strcat(rdBuffer,"Device Error, ");
		if (*standardEventStatusRegister & SNDCSDLM_SE_RANGEERROR)
			strcat(rdBuffer,"Execution Error (Range), ");
		if (*standardEventStatusRegister & SNDCSDLM_SE_SYNTAXERROR)
			strcat(rdBuffer,"Command Error (Syntax), ");
		if (*standardEventStatusRegister & SNDCSDLM_SE_PWRON)
			strcat(rdBuffer,"Power On");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: StandEventEnableQuery                                           */
/* Purpose:  This function returns the instruments standard event enable     */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_StandEventEnableQuery (ViSession      instrSession,
                                                 ViChar _VI_FAR channelName[],
                                                 unsigned char  *standardEventEnableRegister,
                                                 ViChar _VI_FAR statusMessage[])
{
    
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*ESE");
   	strcat(cmdBuffer,channelName);
   	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*standardEventEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits, bits 1,6 are NOT USED */
	if (*standardEventEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*standardEventEnableRegister);
		if (*standardEventEnableRegister & SNDCSDLM_SE_OPC)
			strcat(rdBuffer,"Operation Complete, ");
		if (*standardEventEnableRegister & SNDCSDLM_SE_ERRORQUERY)
			strcat(rdBuffer,"Query Error, ");
		if (*standardEventEnableRegister & SNDCSDLM_SE_DEVICEERROR)
			strcat(rdBuffer,"Device Error, ");
		if (*standardEventEnableRegister & SNDCSDLM_SE_RANGEERROR)
			strcat(rdBuffer,"Execution Error (Range), ");
		if (*standardEventEnableRegister & SNDCSDLM_SE_SYNTAXERROR)
			strcat(rdBuffer,"Command Error (Syntax), ");
		if (*standardEventEnableRegister & SNDCSDLM_SE_PWRON)
			strcat(rdBuffer,"Power On");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtEventQuery                                                  */
/* Purpose:  This function returns the instruments protection event status   */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ProtEventQuery (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          unsigned char  *protEventReg,
                                          ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS");
   	strcat(cmdBuffer,channelName);
   	strcat(cmdBuffer,":PROTECTION:EVENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protEventReg = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protEventReg == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protEventReg);
		if (*protEventReg & SNDCSDLM_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protEventReg & SNDCSDLM_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protEventReg & SNDCSDLM_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protEventReg & SNDCSDLM_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protEventReg & SNDCSDLM_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protEventReg & SNDCSDLM_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protEventReg & SNDCSDLM_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protEventReg & SNDCSDLM_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtEventEnableQuery                                            */
/* Purpose:  This function returns the instruments protection event enable   */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ProtEventEnableQuery (ViSession      instrSession,
                                                ViChar _VI_FAR channelName[],
                                                unsigned char  *protEventEnableRegister,
                                                ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS");
   	strcat(cmdBuffer,channelName);
   	strcat(cmdBuffer,":PROTECTION:ENABLE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protEventEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protEventEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protEventEnableRegister);
		if (*protEventEnableRegister & SNDCSDLM_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protEventEnableRegister & SNDCSDLM_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protEventEnableRegister & SNDCSDLM_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protEventEnableRegister & SNDCSDLM_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protEventEnableRegister & SNDCSDLM_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protEventEnableRegister & SNDCSDLM_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protEventEnableRegister & SNDCSDLM_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protEventEnableRegister & SNDCSDLM_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ProtCondQuery                                                   */
/* Purpose:  This function returns the instruments protection condition      */
/*           status register and a message indicating which bits are active. */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ProtCondQuery (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         unsigned char  *protCondReg,
                                         ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"STATUS");
   	strcat(cmdBuffer,channelName);
   	strcat(cmdBuffer,":PROTECTION:CONDITION?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*protCondReg = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from register bits */
	if (*protCondReg == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*protCondReg);
		if (*protCondReg & SNDCSDLM_PE_VOLTMODE)
			strcat(rdBuffer,"Constant Voltage, ");
		if (*protCondReg & SNDCSDLM_PE_CURRMODE)
			strcat(rdBuffer,"Constant Current, ");
		if (*protCondReg & SNDCSDLM_PE_HW)
			strcat(rdBuffer,"Hardware Fault, ");
		if (*protCondReg & SNDCSDLM_PE_OVP)
			strcat(rdBuffer,"OVP, ");
		if (*protCondReg & SNDCSDLM_PE_OTP)
			strcat(rdBuffer,"OTP, ");
		if (*protCondReg & SNDCSDLM_PE_EXTSD)
			strcat(rdBuffer,"Ext. Shutdown, ");
		if (*protCondReg & SNDCSDLM_PE_FOLD)
			strcat(rdBuffer,"Foldback Mode, ");
		if (*protCondReg & SNDCSDLM_PE_PRGMERROR)
			strcat(rdBuffer,"Programming Error");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: ServiceReqEnableQuery                                           */
/* Purpose:  This function returns the instruments service request enable    */
/*           register and a message indicating which bits are active.        */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ServiceReqEnableQuery (ViSession      instrSession,
                                                 ViChar _VI_FAR channelName[],
                                                 unsigned char  *serviceReqEnableRegister,
                                                 ViChar _VI_FAR statusMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViChar		rdBuffer[BUFFER_SIZE];
    char		*endPtr;

	LOCK
	
	/* build and send SCPI command string */
    strcpy(cmdBuffer,"*SRE");
   	strcat(cmdBuffer,channelName);
   	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* assign value */
	*serviceReqEnableRegister = (unsigned char) strtoul((const char *) rdBuffer, &endPtr, 10);

	/* generate status message from status byte bits, bits 0,3,7 are NOT USED */
	if (*serviceReqEnableRegister == 0)
		strcpy(statusMessage,"00");
	else
	{
		sprintf(rdBuffer,"%02X - ",*serviceReqEnableRegister);
		if (*serviceReqEnableRegister & 0x02)
			strcat(rdBuffer,"Protection Event, ");
		if (*serviceReqEnableRegister & 0x04)
			strcat(rdBuffer,"Error Event Message, ");
		if (*serviceReqEnableRegister & 0x10)
			strcat(rdBuffer,"Message Available, ");
		if (*serviceReqEnableRegister & 0x20)
			strcat(rdBuffer,"Event Status, ");
		if (*serviceReqEnableRegister & 0x40)
			strcat(rdBuffer,"Request Service");
		
		/* remove any ending ',' */
		retCnt = strlen(rdBuffer);
		if (rdBuffer[retCnt - 2] == ',')
			rdBuffer[retCnt - 2] = '\0';

		strcpy(statusMessage,rdBuffer);
	}
	
    UNLOCK
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument to a known state.           */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_reset (ViSession      instrSession,
                                 ViChar _VI_FAR channelName[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
    /*  send SCPI reset command */
    strcpy(cmdBuffer,"*RST");
   	strcat(cmdBuffer,channelName);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    UNLOCK
}

/*===========================================================================*/
/* Function: Clear                                                          */
/* Purpose:  This function clear the instrument error and status info.       */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_clear (ViSession      instrSession,
                                 ViChar _VI_FAR channelName[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI clear command */
    strcpy(cmdBuffer,"*CLS");
   	strcat(cmdBuffer,channelName);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_errorQuery (ViSession      instrSession,
                                      ViPInt32       errorCode,
                                      ViChar _VI_FAR errorMessage[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViUInt32	index;
    ViUInt32	i;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViChar		*cptr;
    ViChar		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
    /*  send SCPI reset command */
    strcpy(cmdBuffer,"SYSTEM:ERROR?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

	/* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* get error code from string */
	sscanf(rdBuffer,"%d",errorCode);
	
	/* get error message, skip ", " in return string */
	cptr  = strchr(rdBuffer,',');
	cptr += 2;
	strcpy(errorMessage,cptr);

	UNLOCK
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_errorMessage (ViSession      instrSession,
                                        ViStatus       statusCode,
                                        ViChar _VI_FAR message[])
{
    ViStatus SNDCSDLM_status,unlockStatus;
    ViInt16 i;
    static SNDCSDLM_tStringValPair statusDescArray[] = {
        {VI_WARN_NSUP_ID_QUERY,                "WARNING: ID Query not supported"},
        {VI_WARN_NSUP_RESET,                   "WARNING: Reset not supported"},
        {VI_WARN_NSUP_SELF_TEST,               "WARNING: Self-test not supported"},
        {VI_WARN_NSUP_ERROR_QUERY,             "WARNING: Error Query not supported"},     
        {VI_WARN_NSUP_REV_QUERY,               "WARNING: Revision Query not supported"},
        {VI_ERROR_PARAMETER1,                  "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,                  "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,                  "ERROR: Parameter 3 out of range"},
        {VI_ERROR_PARAMETER4,                  "ERROR: Parameter 4 out of range"},
        {VI_ERROR_PARAMETER5,                  "ERROR: Parameter 5 out of range"},
        {VI_ERROR_PARAMETER6,                  "ERROR: Parameter 6 out of range"},
        {VI_ERROR_PARAMETER7,                  "ERROR: Parameter 7 out of range"},
        {VI_ERROR_PARAMETER8,                  "ERROR: Parameter 8 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,               "ERROR: Identification query failed"},
        {VI_ERROR_INV_RESPONSE,                "ERROR: Interpreting instrument response"},
        {VI_ERROR_INSTR_FILE_OPEN,             "ERROR: Opening the specified file"},
        {VI_ERROR_INSTR_FILE_WRITE,            "ERROR: Writing to the specified file"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE, "ERROR: Interpreting the instrument's response"},
        
        /* not used, example of user defined Status Code
        {PREFIX_ERROR_INVALID_CONFIGURATION,   "ERROR: Instrument configuration error"},
        */
        
        {VI_NULL, VI_NULL}
    };

    SNDCSDLM_status = viStatusDesc (instrSession, statusCode, message);
    if (SNDCSDLM_status == VI_WARN_UNKNOWN_STATUS) {
        for (i=0; statusDescArray[i].stringName; i++) {
            if (statusDescArray[i].stringVal == statusCode) {
                strcpy (message, statusDescArray[i].stringName);
                return (VI_SUCCESS);
            }
        }
        sprintf (message, "Unknown Error 0x%08lX", statusCode);
        return (VI_WARN_UNKNOWN_STATUS);
    }
    
    UNLOCK
}

/*===========================================================================*/
/* Function: revisionQuery                                                   */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_revisionQuery (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViChar _VI_FAR driverRev[],
                                         ViChar _VI_FAR instrRev[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[BUFFER_SIZE];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"*IDN");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

    /* parse *idn? return string for firware revisions - 4 & 5 fields */
	cptr = rdBuffer;
	for (i = 0; i < 3; ++i)
	{
		cptr = strchr(cptr,',');
		++cptr;
	}
	
	/* copy driver and intrument strings to buffers */
	strcpy(instrRev, cptr);
    strcpy (driverRev, SNDCSDLM_REVISION);

	UNLOCK
}

/*===========================================================================*/
/* Function: Identify Query                                                  */
/* Purpose:  This function returns the *idn? response.                       */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_idQuery (ViSession      instrSession,
                                   ViChar _VI_FAR channelName[],
                                   ViChar _VI_FAR idBuffer[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"*IDN");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy id string to buffer */
	strcpy(idBuffer, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ModelQuery                                                   */
/* Purpose:  This function reads the instrument's Maximum Voltage and        */
/*           Current for the specified channel.                              */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ModelQuery    (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViReal64       *modelVolt,
                                         ViReal64       *modelCurr)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    *modelVolt = 0.0;
    *modelCurr = 0.0;

    /* build and send SCPI command string - voltage */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":MODEL:VOLTAGE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *modelVolt = atof(rdBuffer);

    /* build and send SCPI command string - current */
    strcpy(cmdBuffer,"CALIBRATE");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":MODEL:CURRENT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *modelCurr = atof(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_close (ViSession instrSession)
{
    SNDCSDLM_instrRange instrPtr;
    ViSession rmSession;
    ViStatus SNDCSDLM_status;

    if ((SNDCSDLM_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return SNDCSDLM_status;
 	if ((SNDCSDLM_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return SNDCSDLM_status;
            
    if (instrPtr != NULL) 
        free (instrPtr);
    
    SNDCSDLM_status = viClose (instrSession);
    viClose (rmSession);

	return SNDCSDLM_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViBooleanRange (ViBoolean val)
{
    return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Unsigned Char Value Out Of Range - ViUInt8                      */
/* Purpose:  This function checks an unsigned char value to see if it        */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViUInt8Range (ViUInt8 val, ViUInt8 min, ViUInt8 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean SNDCSDLM_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the SNDCSDLM_init function.  When   */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from SNDCSDLM_init to       */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus SNDCSDLM_initCleanUp (ViSession  openRMSession,
                               ViPSession openInstrSession,
                               ViStatus   currentStatus)
{
    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}


// TCP/IP commands 6-28-05 // RBF:


/*===========================================================================*/
/* Function: ConfigIp                                                        */
/* Purpose:  This function sets the IP address to the specified value.       */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigIp (ViSession       instrSession,
	                                ViChar _VI_FAR  channelName[],
	                                ViChar _VI_FAR  ipSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:IP ");
    strcat(cmdBuffer,ipSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the IP address.                             */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_IpQuery (ViSession      instrSession,
                                   ViChar _VI_FAR channelName[],
                                   ViChar _VI_FAR ipGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:IP");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(ipGetAddress, rdBuffer);

    UNLOCK
}
                                                  
/*===========================================================================*/
/* Function: ConfigMask                                                      */
/* Purpose:  This function sets the mask address to the specified value.     */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigMask (ViSession      instrSession,
	                                  ViChar _VI_FAR channelName[],
	                                  ViChar _VI_FAR maskSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:MASK ");
    strcat(cmdBuffer,maskSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the Mask address.                           */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_MaskQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViChar _VI_FAR maskGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:MASK");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(maskGetAddress, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigGate                                                      */
/* Purpose:  This function sets the Gate address to the specified value.     */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigGate (ViSession       instrSession,
	                                  ViChar _VI_FAR  channelName[],
	                                  ViChar _VI_FAR  gateSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:GATE ");
    strcat(cmdBuffer,gateSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: GateQuery                                                       */
/* Purpose:  This function reads the Gate address    .                       */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_GateQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViChar _VI_FAR gateGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:GATE");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(gateGetAddress, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDNS                                                       */
/* Purpose:  This function sets the DNS address to the specified value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigDNS (ViSession       instrSession,
	                                 ViChar _VI_FAR  channelName[],
	                                 ViChar _VI_FAR  dnsSetAddress[])
{
   	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DNS ");
    strcat(cmdBuffer,dnsSetAddress);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: DNSQuery                                                        */
/* Purpose:  This function reads the DNS address.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_DNSQuery (ViSession      instrSession,
                                    ViChar _VI_FAR channelName[],
                                    ViChar _VI_FAR dnsGetAddress[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:DNS");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(dnsGetAddress, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigPort                                                      */
/* Purpose:  This function sets the listen port to the specified value.      */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigPort (ViSession      instrSession,
                                      ViChar _VI_FAR channelName[],
                                      ViInt32       listenPort)
{
   ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:PORT ");
    sprintf(tmpBuffer,"%d",listenPort);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the listen port.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_PortQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViInt32        *listenPort)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - listen port */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:PORT?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *listenPort = atoi(rdBuffer);

    UNLOCK
}

                                                  
/*===========================================================================*/
/* Function: ConfigTerm                                                      */
/* Purpose:  This function sets the terminator attribute					 */
/*           to the specified value.                                         */
/*           1 = CR, 2 = LF, 3 = CR LF, 4 = LF CR  							 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigTerm (ViSession      instrSession,
                                      ViChar _VI_FAR channelName[],
                                      ViInt32        terminator)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:TERM ");
    sprintf(tmpBuffer,"%d",terminator);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the terminator attribute.                   */
/*           1 = CR, 2 = LF, 3 = CR LF, 4 = LF CR  							 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_TermQuery (ViSession      instrSession,
                                     ViChar _VI_FAR channelName[],
                                     ViInt32        *terminator)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:TERM?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
	if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *terminator = atof(rdBuffer);

	UNLOCK
}

/*===========================================================================*/
/* Function: ConfigSlaveBaud                                                 */
/* Purpose:  This function sets the Slave Baud Rate to the specified value.  */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigSlaveBaud (ViSession      instrSession,
                                           ViChar _VI_FAR channelName[],
                                           ViReal64        slaveBaud)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":SLAVEBAUD ");
    sprintf(tmpBuffer,"%f",slaveBaud);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the Slave Baud Rate address.                */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_SlaveBaudQuery (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViReal64       *slaveBaud)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":SLAVEBAUD?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
	if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *slaveBaud = atof(rdBuffer);

	UNLOCK
}

/*===========================================================================*/
/* Function: ConfigHost                                                        */
/* Purpose:  This function sets the Host information with 					 */
/*           the specified string. 										     */
/*			 Note: 15 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigHost             (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR SetHost[])
{
  	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:HOST ");
    strcat(cmdBuffer,SetHost);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: hostQuery                                                         */
/* Purpose:  This function reads the Host information string.                */
/*			 Note: 15 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_HostQuery              (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR getHost[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:HOST");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(getHost, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDesc                                                      */
/* Purpose:  This function sets the desacription information with 		     */
/*           the specified string. 										     */
/*			 Note: 64 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigDesc             (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR setDesc[])
{
  	ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];
    
	LOCK
	
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DESC ");
    strcat(cmdBuffer,setDesc);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

    UNLOCK
}

/*===========================================================================*/
/* Function: IpQuery                                                         */
/* Purpose:  This function reads the decription string.                      */
/*			 Note: 64 character alphanumeric string							 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_DescQuery              (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR getDesc[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:DESC");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(getDesc, rdBuffer);

    UNLOCK
}
                                                  
/*===========================================================================*/
/* Function: MACQuery                                                        */
/* Purpose:  This function reads the MAC address.                            */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_MACQuery               (ViSession      instrSession,
                                                  ViChar _VI_FAR channelName[],
                                                  ViChar _VI_FAR getMAC[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"SYST");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":NET:MAC");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(getMAC, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigAutoIp                                                    */
/* Purpose:  This function sets the Auto IP address mode.                    */
/*           0 = Disable Auto IP, 1 = Enable Auto IP						 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigAutoIP (ViSession      instrSession,
                                        ViChar _VI_FAR channelName[],
                                        ViBoolean      autoIPMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:AUTOIP ");
    sprintf(tmpBuffer,"%d",autoIPMode);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: AutoIpQuery                                                     */
/* Purpose:  This function reads the Auto IP address mode.                   */
/*           0 = Disable Auto IP, 1 = Enable Auto IP						 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_AutoIPQuery (ViSession      instrSession,
                                       ViChar _VI_FAR channelName[],
                                       ViBoolean      *autoIPMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:AUTOIP?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *autoIPMode = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigDHCPMode                                                  */
/* Purpose:  This function sets the DHCP Mode to the specified value.        */
/*           0 = Static IP address, 1 =  Use DHCP						  	 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigDHCPMode (ViSession      instrSession,
                                          ViChar _VI_FAR channelName[],
                                          ViBoolean      dhcpMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DHCPMODE ");
    sprintf(tmpBuffer,"%d",dhcpMode);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: DHCPModeQuery                                                   */
/* Purpose:  This function reads the DHCp Mode.                              */
/*           0 = Static IP address, 1 =  Use DHCP						  	 */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_DHCPModeQuery (ViSession      instrSession,
                                         ViChar _VI_FAR channelName[],
                                         ViBoolean      *dhcpMode)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:DHCPMODE?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *dhcpMode = atoi(rdBuffer);

    UNLOCK
}
                                                  
/*===========================================================================*/
/* Function: ConfigLANLED                                                    */
/* Purpose:  This function sets the LANLED Mode to the specified value.      */
/*           1 = BLINKON, 0 = BLINKOFF                    				     */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigLANLED (ViSession      instrSession,
                                        ViChar _VI_FAR channelName[],
                                        ViBoolean      lanLED)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

    LOCK
    
    /* build and send SCPI command string */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:LANLED ");
    if(lanLED)
    {
    	strcat(cmdBuffer,"BLINKON");   
    }
    else
    {
		strcat(cmdBuffer,"BLINKOFF");       
    }
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: LANLEDQuery                                                     */
/* Purpose:  This function reads the  LANLED Mode.                           */
/*           1 = BLINKON, 0 = BLINKOFF                    				     */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_LANLEDQuery (ViSession      instrSession,
                                       ViChar _VI_FAR channelName[],
                                       ViBoolean      *lanLED)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		rdBuffer[BUFFER_SIZE];

	LOCK
	
    /* build and send SCPI command string - voltage limit */
    strcpy(cmdBuffer,"SYST");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":NET:LANLED?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK
        
    /* get returned query string */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';
    
    *lanLED = atoi(rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigLastCalDate                                               */
/* Purpose:  This function sets the Last Calibration Date                    */
/*           to the specified value.                                         */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigLastCalDate (ViSession      instrSession,
	                                         ViChar _VI_FAR channelName[],
	                                         ViReal64       month,
	                                         ViReal64       day,
	                                         ViReal64       year)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"CAL");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":MOD:LASTCALDATE ");
    sprintf(tmpBuffer,"%f",month);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",day);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",year);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: LastCalDateQuery                                                */
/* Purpose:  This function reads the Last Calibration Date.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_LastCalDateQuery (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViChar _VI_FAR lastCalDate[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"CAL");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":MOD:LASTCALDATE");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(lastCalDate, rdBuffer);

    UNLOCK
}

/*===========================================================================*/
/* Function: ConfigNextCalDate                                               */
/* Purpose:  This function sets the Next Calibration Date                    */
/*           to the specified value.                                         */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_ConfigNextCalDate (ViSession      instrSession,
                                             ViChar _VI_FAR channelName[],
                                             ViReal64       month,
                                             ViReal64       day,
                                             ViReal64       year)
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViByte		cmdBuffer[80];
    ViByte		tmpBuffer[80];

	LOCK

    /* build and send SCPI command string */
    strcpy(cmdBuffer,"CAL");
    strcat(cmdBuffer,channelName);
    strcat(cmdBuffer,":MOD:NEXTCALDATE ");
    sprintf(tmpBuffer,"%f",month);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",day);
    strcat(cmdBuffer,tmpBuffer);
    strcat(cmdBuffer," ");
    sprintf(tmpBuffer,"%f",year);
    strcat(cmdBuffer,tmpBuffer);
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
    sendCnt = strlen(cmdBuffer);
    SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt);

	UNLOCK
}

/*===========================================================================*/
/* Function: NextCalDateQuery                                                */
/* Purpose:  This function reads the Next Calibration Date.                  */
/*===========================================================================*/
ViStatus CVIFUNC SNDCSDLM_NextCalDateQuery (ViSession      instrSession,
                                            ViChar _VI_FAR channelName[],
                                            ViChar _VI_FAR nextCalDate[])
{
    ViUInt32	retCnt = 0;
    ViUInt32	sendCnt = 0;
    ViStatus	SNDCSDLM_status,unlockStatus;
    ViInt16		i;
    ViChar		*cptr;
    ViByte		rdBuffer[BUFFER_SIZE];
    ViByte		cmdBuffer[80];

	LOCK
	
	/* send SCPI identify command */
	strcpy(cmdBuffer,"CAL");
	strcat(cmdBuffer,channelName);
	strcat(cmdBuffer,":MOD:NEXTCALDATE");
	strcat(cmdBuffer,"?");
    /* add <CR> to end of command for ENET_IF */
    if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	sendCnt = strlen(cmdBuffer);
    if ((SNDCSDLM_status = viWrite (instrSession, cmdBuffer, sendCnt, &retCnt)) < 0)
        UNLOCK

    /* get returned query */
    if ((SNDCSDLM_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
        UNLOCK
	rdBuffer[retCnt] = '\0';

	/* copy ip string to buffer */
	strcpy(nextCalDate, rdBuffer);

    UNLOCK
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
